unit Glassimg;

{
Name:					TGlassImage
Description:  Image control for Delphi16 with selectable transparent color
Author:				Richard Shotbolt 100327.2305@compuserve.com

TGlassImage is a freeware transparent image control. It has one extra property,
GlassColor, which defines the transparent color.
}

interface

uses
  WinTypes, WinProcs, SysUtils, Graphics, Classes, Controls, Menus, Consts;

type

	TGlassImage = class(TGraphicControl)
  private
    FPicture: TPicture;
    FAutoSize: Boolean;
    FStretch: Boolean;
    FCenter: Boolean;
    FReserved: Byte;
    FImageBitmap: TBitmap;
    FAndBitmap: TBitmap;
    FOrBitmap: TBitmap;
    FGlassColor: TColor;
    MasksMade: Boolean;
    function GetCanvas: TCanvas;
    procedure PictureChanged(Sender: TObject);
    procedure SetAutoSize(Value: Boolean);
    procedure SetCenter(Value: Boolean);
    procedure SetPicture(Value: TPicture);
    procedure SetStretch(Value: Boolean);
    procedure SetGlassColor(Value: TColor);
  protected
    function GetPalette: HPALETTE; override;
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Canvas: TCanvas read GetCanvas;
  published
    property Align;
    property AutoSize: Boolean read FAutoSize write SetAutoSize default False;
    property Center: Boolean read FCenter write SetCenter default False;
    property DragCursor;
    property DragMode;
    property Enabled;
    property ParentShowHint;
    property Picture: TPicture read FPicture write SetPicture;
    property PopupMenu;
    property ShowHint;
    property Stretch: Boolean read FStretch write SetStretch default False;
    property GlassColor: TColor read FGlassColor write SetGlassColor default clBtnFace;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;


procedure Register;

implementation

{ TGlassImage }

constructor TGlassImage.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FPicture := TPicture.Create;
  FPicture.OnChange := PictureChanged;
  FAndBitmap := TBitmap.Create;
  FOrBitmap := TBitmap.Create;
  FGlassColor := clBtnFace;
  Height := 105;
  Width := 105;
end;

destructor TGlassImage.Destroy;
begin
	FOrBitmap.Free;
  FAndBitmap.Free;
  FPicture.Free;
  inherited Destroy;
end;

function TGlassImage.GetPalette: HPALETTE;
begin
  Result := 0;
  if FPicture.Graphic is TBitmap then
    Result := TBitmap(FPicture.Graphic).Palette;
end;

procedure TGlassImage.Paint;
var
  Dest: TRect;
  OldMode: TCopyMode;
  OldPal: HPALETTE;
begin
{ Get bounding rectangle }
if FStretch then
	Dest := GetClientRect
else if FCenter then
  Dest := Bounds((Width - Picture.Width) div 2, (Height - Picture.Height) div 2,
    Picture.Width, Picture.Height)
else
  Dest := Rect(0, 0, Picture.Width, Picture.Height);
if (not MasksMade) then
	begin
  { Recreate mask bitmaps }
  with FOrBitmap do
  	begin
    { Default image color to FGlassColor }
    Canvas.Brush.Color := FGlassColor;
    Height := Dest.Bottom - Dest.Top;
    Width := Dest.Right - Dest.Left;
    Canvas.FillRect(Dest);
    Canvas.CopyMode := cmSrcCopy;
    { Copy image to FOrBitmap, sizing it at the same time }
    Canvas.StretchDraw(Dest, Picture.Graphic);
    { Copy the resized image to FAndBitmap as well }
  	FAndBitmap.Assign(FOrBitmap);
  	Canvas.Brush.Color := clBlack;
  { Change all transparent color pixels in OR bitmap to black }
  	Canvas.BrushCopy(Dest, FOrBitmap, Dest, FGlassColor);
    end;
  with FAndBitmap.Canvas do
  	begin
  	{ Change all transparent color pixels in AND bitmap to white }
  	Brush.Color := clWhite;
  	BrushCopy(Dest, FAndBitmap, Dest, FGlassColor);
  	{ XOR the AND and OR bitmaps into the AND bitmap }
  	{ All pixels in transparent color will be white  }
  	{ All other pixels will be black }
  	CopyMode := cmSrcInvert;
  	CopyRect(Dest, FOrBitmap.Canvas, Dest);
    end;
  MasksMade := True;
  end;
with inherited Canvas do begin
    OldMode := CopyMode;
    try
      { Realize the image's palette into the destination canvas }
      if GetPalette <> 0 then
      	begin
        OldPal := SelectPalette(Canvas.Handle, GetPalette, False);
        RealizePalette(Canvas.Handle);
				end;
      try
      	{ Make a black hole in the canvas }
        CopyMode := cmSrcAnd;
        StretchDraw(Dest, FAndBitmap);
        { Fill the black hole with the parts of the image not in the transparent color }
        CopyMode := cmSrcPaint;
        StretchDraw(Dest, FOrBitmap);
        { Draw dashed rectangle around control at design time }
        if (csDesigning in ComponentState) then
          begin
          	Pen.Style := psDash;
            Brush.Style := bsClear;
            Rectangle(0, 0, Width, Height);
          end;
      finally
      	{ Reset palette }
        if GetPalette <> 0 then
          SelectPalette(Canvas.Handle, OldPal, False);
      end;
    finally
      CopyMode := OldMode;
    end;
  end;
end;

function TGlassImage.GetCanvas: TCanvas;
var
  Bitmap: TBitmap;
begin
  if Picture.Graphic = nil then
  begin
    Bitmap := TBitmap.Create;
    try
      Bitmap.Width := Width;
      Bitmap.Height := Height;
      Picture.Graphic := Bitmap;
    finally
      Bitmap.Free;
    end;
  end;
  if Picture.Graphic is TBitmap then
    Result := TBitmap(Picture.Graphic).Canvas
  else
    raise EInvalidOperation.Create(LoadStr(SImageCanvasNeedsBitmap));
end;

procedure TGlassImage.SetAutoSize(Value: Boolean);
begin
  FAutoSize := Value;
  PictureChanged(Self);
end;

procedure TGlassImage.SetCenter(Value: Boolean);
begin
  if FCenter <> Value then
  begin
    FCenter := Value;
    MasksMade := False;
    Invalidate;
  end;
end;

procedure TGlassImage.SetPicture(Value: TPicture);
begin
  FPicture.Assign(Value);
end;

procedure TGlassImage.SetStretch(Value: Boolean);
begin
  FStretch := Value;
  MasksMade := False;
  Invalidate;
end;

procedure TGlassImage.PictureChanged(Sender: TObject);
begin
  if AutoSize and (Picture.Width > 0) and (Picture.Height > 0) then
    SetBounds(Left, Top, Picture.Width, Picture.Height);
  if (Picture.Graphic is TBitmap) and (Picture.Width = Width) and
    (Picture.Height = Height) then
    ControlStyle := ControlStyle + [csOpaque] else
    ControlStyle := ControlStyle - [csOpaque];
  MasksMade := False;
  Invalidate;
  Parent.Refresh;
end;

procedure TGlassImage.SetGlassColor(Value: TColor);
begin
  if Value <> FGlassColor then
    begin
      FGlassColor := Value;
      MasksMade := False;
      Invalidate;
    end;
end;

{ Register TGlassImage on the 'Samples' page }
procedure Register;
begin
	RegisterComponents('Samples', [TGlassImage]);
end;

end.
